/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/***********************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Onset of Civil Conflict in Repeated Cross-Country Data - Robustness to Accounting for Commodity Export Price Shocks */
/***********************************************************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_onset_xpshk.log", text replace ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/******************************************************************************************************/
/* Generate 1-year lags for the time-varying covariates and for conflict incidence and peace duration */
/******************************************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 cconf_pri1 tap_a_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/**************************************************************************************************/
/* Generate cubic splines to control for the dependence of civil conflict onset on peace duration */
/**************************************************************************************************/

mkspline spline_pri1_  = tap_a_pri1_l1, cubic nknots(4) ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.year ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable iscol_gbr_l1 "U.K. colony dummy, lagged" ;
label variable iscol_fra_l1 "French colony dummy, lagged" ;
label variable iscol_oth_l1 "Other colony dummy, lagged" ;
label variable legor_uk     "British legal origin dummy" ;
label variable legor_fr     "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Executive constraints, lagged" ;
label variable ddemoc_p4_v17_l1 "Democracy dummy, lagged" ;
label variable dautoc_p4_v17_l1 "Autocracy dummy, lagged" ;

/* Lagged conflict incidence */
label variable cconf_pri1_l1 "Civil conflict incidence, lagged" ;

/* Commodity export price shock variables */
label variable pshock_npi_p            "Aggregate price shock" ;
label variable pshock_npi_p1           "Aggregate price shock, lagged" ;
label variable pshock_npi_p2           "Aggregate price shock, twice lagged" ;
label variable pshock_annual_npi_p     "Annual crop price shock" ;
label variable pshock_annual_npi_p1    "Annual crop price shock, lagged" ;
label variable pshock_annual_npi_p2    "Annual crop price shock, twice lagged" ;
label variable pshock_perennial_npi_p  "Perennial crop price shock" ;
label variable pshock_perennial_npi_p1 "Perennial crop price shock, lagged" ;
label variable pshock_perennial_npi_p2 "Perennial crop price shock, twice lagged" ;
label variable pshock_extract_npi_p    "Extractive crop price shock" ;
label variable pshock_extract_npi_p1   "Extractive crop price shock, lagged" ;
label variable pshock_extract_npi_p2   "Extractive crop price shock, twice lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global splines "spline_pri1_1 spline_pri1_2 spline_pri1_3" ;
global timedum "_Iyear_1960-_Iyear_2006" ;
global shocks1 "pshock_npi_p pshock_npi_p1 pshock_npi_p2" ;
global shocks2 "pshock_annual_npi_p pshock_annual_npi_p1 pshock_annual_npi_p2 pshock_perennial_npi_p pshock_perennial_npi_p1 pshock_perennial_npi_p2 pshock_extract_npi_p pshock_extract_npi_p1 pshock_extract_npi_p2" ;

/***************************************************/
/* Generate a sample indicator for the regressions */
/***************************************************/

qui egen smpl_flag1 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks1} ${shocks2}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & year >= 1960 & year <= 2007) ;

qui egen smpl_flag2 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks1} ${shocks2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & year >= 1960 & year <= 2007) ;

/******************************************/
/* Run the regressions for conflict onset */
/******************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks1} ${contold} ${splines} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col1_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks1} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contold} ${splines} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col2_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks2} ${contold} ${splines} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col3_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${shocks2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contold} ${splines} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col4_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks1} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col5_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks1} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks1} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col6_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks1} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks2} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col7_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks2} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col8_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${shocks2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/stab_onset_xpshk.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) drop(pdiv_aa:*, relax) keep(pdiv_aa ${shocks1} ${shocks2}, relax) 
  order(pdiv_aa ${shocks1} ${shocks2}) 
  indicate("Continent dummies=${contall}" 
           "Time dummies=_Iyear_*" 
           "Controls for temporal spillovers=cconf_pri1_l1 ${splines}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p, fmt(%9.0fc %9.0f a2) labels("Observations" "Countries" "Pseudo \$R^2\$") layout(@ @ @)) 
  varwidth(41) msign("\$-\$") label 
  prehead("\begin{tabular*}{620pt}{@{\extracolsep{\fill}}lcccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{4}{c}{Old World}&\multicolumn{4}{c}{Global}\\" 
          "\cmidrule(r){2-5}\cmidrule(l){6-9}") 
  numbers mlabels("Probit" "Probit" "Probit" "Probit" "IV Probit" "IV Probit" "IV Probit" "IV Probit") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{8}{c}{Annual PRIO25 civil conflict onset, 1960--2007}\\" 
           "\cmidrule{2-9}") 
  eqlabels("", none) varlabels(,elist(pshock_extract_npi_p2 \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me using "../../results/tables/stab_onset_xpshk.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(41) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity", elist(pdiv_aa \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/stab_onset_xpshk.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(41) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
