/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*************************************************************************************************************/
/* TABLE: Ethnic Fractionalization, Polarization, and the Frequency of Civil Conflict Onset across Countries */
/*************************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_freqy_fprob.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

gen lnnum_ccst_60_17_avg_pri1 = ln(1 + num_ccst_60_17_avg_pri1) ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat  "Absolute latitude" ;
label variable ruggavg "Ruggedness" ;
label variable elevavg "Mean elevation" ;
label variable elevrng "Range of elevation" ;
label variable suitavg "Mean land suitability" ;
label variable suitrng "Range of land suitability" ;
label variable distcr  "Distance to nearest waterway" ;
label variable island  "Island nation dummy" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global contall "africa asia namerica samerica oceania" ;

/***************************************************/
/* Generate a sample indicator for the regressions */
/***************************************************/

qui egen smpl_flag = rowmiss(lnnum_ccst_60_17_avg_pri1 efrac des_pol15 ${geovars}) ;
qui replace smpl_flag = (smpl_flag == 0) ;

/***********************/
/* Run the regressions */
/***********************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: reg lnnum_ccst_60_17_avg_pri1 efrac if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col1 ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: reg lnnum_ccst_60_17_avg_pri1 efrac ${geovars} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: reg lnnum_ccst_60_17_avg_pri1 efrac ${geovars} ${contall} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col3 ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: reg lnnum_ccst_60_17_avg_pri1 des_pol15 if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col4 ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg lnnum_ccst_60_17_avg_pri1 des_pol15 ${geovars} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: reg lnnum_ccst_60_17_avg_pri1 des_pol15 ${geovars} ${contall} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: reg lnnum_ccst_60_17_avg_pri1 efrac des_pol15 if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: reg lnnum_ccst_60_17_avg_pri1 efrac des_pol15 ${geovars} if smpl_flag == 1, vce(robust) first noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col8 ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: reg lnnum_ccst_60_17_avg_pri1 efrac des_pol15 ${geovars} ${contall} if smpl_flag == 1, vce(robust) first noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col9 ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/stab_freqy_fprob.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(efrac des_pol15, relax) 
  order(efrac des_pol15) 
  indicate("Continent dummies=${contall}" "Controls for geography=${geovars}", labels("\$\times\$" " ")) 
  stats(N adjr2, fmt(%9.0f a2) labels("Observations" "Adjusted \$R^2\$") layout(@ @)) 
  varwidth(28) msign("\$-\$") label 
  prehead("\begin{tabular*}{565pt}{@{\extracolsep{\fill}}lccccccccc}" "\toprule") 
  numbers mlabels("OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{9}{c}{Log number of new PRIO25 civil conflict onsets per year, 1960--2017}\\" 
           "\cmidrule{2-10}") 
  varlabels(, elist(des_pol15 \addlinespace)) 
  prefoot("\midrule") postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
