/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*******************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Frequency of Civil Conflict Onset across Countries - Robustness to Accounting for Ethnic and Spatial Inequality */
/*******************************************************************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_freqy_egini.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

gen lnnum_ccst_60_17_avg_pri1 = ln(1 + num_ccst_60_17_avg_pri1) ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable evercol_gbr "Ever a U.K. colony dummy" ;
label variable evercol_fra "Ever a French colony dummy" ;
label variable evercol_oth "Ever a non-U.K./non-French colony dummy" ;
label variable legor_uk    "British legal origin dummy" ;
label variable legor_fr    "French legal origin dummy" ;

/* Contemporary institutional variables */
label variable xconst_1960_2017_avg_p4_v17 "Executive constraints, 1960--2017 average" ;
label variable democ_1960_2017_shr_p4_v17  "Fraction of years under democracy, 1960--2017" ;
label variable autoc_1960_2017_shr_p4_v17  "Fraction of years under autocracy, 1960--2017" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet                 "Oil or gas reserve discovery" ;
label variable lnpop_1960_2017_avg_wdi          "Log population, 1960--2017 average" ;
label variable lngdppc_cu_usd_1960_2017_avg_wdi "Log GDP per capita, 1960--2017 average" ;

/* Ethnic and spatial inequality variables */
label variable grg_a_lum00pc "Ethnic inequality in luminosity" ;
label variable v1_a_lum00pc  "Spatial inequality in luminosity" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "evercol_gbr evercol_fra evercol_oth" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_1960_2017_avg_p4_v17" ;
global polregs "democ_1960_2017_shr_p4_v17 autoc_1960_2017_shr_p4_v17" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_1960_2017_avg_wdi" ;
global ypercap "lngdppc_cu_usd_1960_2017_avg_wdi" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global ethineq "grg_a_lum00pc v1_a_lum00pc" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0) ;

qui egen smpl_flag2 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0) ;

/***********************/
/* Run the regressions */
/***********************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col1 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col1_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col2 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col2_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col3 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col3_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col3 ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col4 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col4_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col4 ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col5 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col5_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${contold} if smpl_flag1 == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col6 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col6_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${contold} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if smpl_flag2 == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col7 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col7_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivregress 2sls lnnum_ccst_60_17_avg_pri1 ${ethineq} ${geovars} ${contall} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(robust) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col8_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: ivregress 2sls lnnum_ccst_60_17_avg_pri1 ${ethineq} ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(robust) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col9 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col9_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/stab_freqy_egini.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa ${ethineq}, relax) 
  order(pdiv_aa ${ethineq}) 
  indicate("Continent dummies=${contall}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N pc_pdiv adjr2, fmt(%9.0f a2 a2) 
        labels("Observations" 
               "Partial \$R^2\$ of population diversity" 
               "Adjusted \$R^2\$") layout(@ @ @)) 
  varwidth(44) msign("\$-\$") label 
  prehead("\begin{tabular*}{655pt}{@{\extracolsep{\fill}}lccccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{5}{c}{Global}&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}\\" 
          "\cmidrule(r){2-6}\cmidrule(lr){7-8}\cmidrule(l){9-10}") 
  numbers mlabels("OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "2SLS" "2SLS") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{9}{c}{Log number of new PRIO25 civil conflict onsets per year, 1960--2017}\\" 
           "\cmidrule{2-10}") 
  varlabels(, elist(v1_a_lum00pc \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me col9_me using "../../results/tables/stab_freqy_egini.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_1090 "Effect of 10th--90th \%ile move in diversity", elist(me_1090 \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/stab_freqy_egini.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
