/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*******************************************************************************************************/
/* TABLE: Precolonial Population Diversity and the Occurrence of Historical Conflicts across Countries */
/*******************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/mtab_histo.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

foreach var of varlist numconf*_brecke { ;
  gen ln`var' = ln(1 + `var') ;
} ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variable */
label variable pdiv "Population diversity (precolonial)" ;

/* Geographical variables */
label variable abslat  "Absolute latitude" ;
label variable ruggavg "Ruggedness" ;
label variable elevavg "Mean elevation" ;
label variable elevrng "Range of elevation" ;
label variable suitavg "Mean land suitability" ;
label variable suitrng "Range of land suitability" ;
label variable distcr  "Distance to nearest waterway" ;
label variable island  "Island nation dummy" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global regions "wb_ssa wb_mena wb_eca wb_sas" ;

/*****************************************************/
/* Generate the sample indicator for the regressions */
/*****************************************************/

qui egen smpl_flag = rowmiss(lnnumconf*_brecke pdiv ${geovars}) ;
qui replace smpl_flag = (smpl_flag == 0) ;

/*********************************************************/
/* Run the regressions for the number of conflict onsets */
/*********************************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: reg lnnumconfall_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numconfall_brecke if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv if e(sample) == 1, percentiles(10 90) ;
eststo col1_me: nlcom (me_1090: _b[pdiv] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumconfall_brecke pdiv ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv"), 1]^2 : col1 ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: reg lnnumconf15_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numconf15_brecke if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv if e(sample) == 1, percentiles(10 90) ;
eststo col2_me: nlcom (me_1090: _b[pdiv] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumconf15_brecke pdiv ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv"), 1]^2 : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: reg lnnumconf16_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numconf16_brecke if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv if e(sample) == 1, percentiles(10 90) ;
eststo col3_me: nlcom (me_1090: _b[pdiv] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumconf16_brecke pdiv ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv"), 1]^2 : col3 ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: reg lnnumconf17_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numconf17_brecke if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv if e(sample) == 1, percentiles(10 90) ;
eststo col4_me: nlcom (me_1090: _b[pdiv] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumconf17_brecke pdiv ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv"), 1]^2 : col4 ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg lnnumconf18_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numconf18_brecke if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv if e(sample) == 1, percentiles(10 90) ;
eststo col5_me: nlcom (me_1090: _b[pdiv] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumconf18_brecke pdiv ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv"), 1]^2 : col5 ;

/*****************************************************/
/* Run the regressions for the onset of any conflict */
/*****************************************************/

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: probit anyconfall_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui margins, predict(pr) at((p10) pdiv) at((p90) pdiv) vce(unconditional) vsquish post ;
eststo col6_me: nlcom (me_1090: _b[2._at] - _b[1._at]), post ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: probit anyconf15_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui margins, predict(pr) at((p10) pdiv) at((p90) pdiv) vce(unconditional) vsquish post ;
eststo col7_me: nlcom (me_1090: _b[2._at] - _b[1._at]), post ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: probit anyconf16_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui margins, predict(pr) at((p10) pdiv) at((p90) pdiv) vce(unconditional) vsquish post ;
eststo col8_me: nlcom (me_1090: _b[2._at] - _b[1._at]), post ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: probit anyconf17_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui margins, predict(pr) at((p10) pdiv) at((p90) pdiv) vce(unconditional) vsquish post ;
eststo col9_me: nlcom (me_1090: _b[2._at] - _b[1._at]), post ;

/* --------- */
/* COLUMN 10 */
/* --------- */
eststo col10: probit anyconf18_brecke pdiv ${geovars} ${regions} if smpl_flag == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui margins, predict(pr) at((p10) pdiv) at((p90) pdiv) vce(unconditional) vsquish post ;
eststo col10_me: nlcom (me_1090: _b[2._at] - _b[1._at]), post ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 col10 using "../../results/tables/mtab_histo.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv) 
  indicate("Region dummies=${regions}" "Controls for geography=${geovars}", labels("\$\times\$" " ")) 
  stats(N pc_pdiv r2_a r2_p, fmt(%9.0f a2 a2 a2) 
        labels("Observations" 
               "Partial \$R^2\$ of population diversity" 
               "Adjusted \$R^2\$" 
               "Pseudo \$R^2\$") layout(@ @ @ @)) 
  varwidth(44) msign("\$-\$") label 
  prehead("\begin{tabular*}{820pt}{@{\extracolsep{\fill}}lcccccccccc}" 
          "\toprule" 
          "Historical period:&1400--1799&1400--1499&1500--1599&1600--1699&1700--1799&1400--1799&1400--1499&1500--1599&1600--1699&1700--1799\\" 
          "\cmidrule(r){2-2}\cmidrule(lr){3-3}\cmidrule(lr){4-4}\cmidrule(lr){5-5}\cmidrule(lr){6-6}\cmidrule(lr){7-7}\cmidrule(lr){8-8}\cmidrule(lr){9-9}\cmidrule(lr){10-10}\cmidrule(l){11-11}") 
  numbers mlabels("OLS" "OLS" "OLS" "OLS" "OLS" "Probit" "Probit" "Probit" "Probit" "Probit") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{5}{c}{Number of conflict onsets in historical period}&\multicolumn{5}{c}{Onset of any conflict in historical period}\\" 
           "\cmidrule(r){2-6}\cmidrule(l){7-11}") 
  eqlabels("", none) varlabels(,elist(pdiv \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me col9_me col10_me using "../../results/tables/mtab_histo.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_1090 "Effect of 10th--90th \%ile move in diversity") 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
