/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*********************************************************************************************************************/
/* FIGURE: Population Diversity and Proximate Determinants of the Frequency of Civil Conflict Onset across Countries */
/*********************************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/**************************/
/* Specify a color scheme */
/**************************/

global RGBcolor1 "215 48 39" ;
global RGBcolor2 "252 141 89" ;
global RGBcolor3 "254 224 144" ;
global RGBcolor4 "224 243 248" ;
global RGBcolor5 "145 191 219" ;
global RGBcolor6 "69 117 180" ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/xfig_mechs.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

gen lnnum_ccst_60_17_avg_pri1 = ln(1 + num_ccst_60_17_avg_pri1) ;
gen lnnumgrps_fea03 = ln(1 + numgrps_fea03) ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global contall "africa asia namerica samerica oceania" ;
global regions "wb_ssa wb_lac" ;

/********************************************************/
/* Generate the sample indicator(s) for the regressions */
/********************************************************/

qui egen smpl_flag1 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} lnnumgrps_fea03) ;
qui egen smpl_flag2 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} wvs_trust) ;
qui egen smpl_flag3 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} wvs_leftright_sd) ;

/* For regressions incorporating WVS measures, ignore "Philippines" and "Dominican Republic" since WVS measures are noisy for these observations */
replace smpl_flag2 = . if wbcode == "PHL" | wbcode == "DOM" ;
replace smpl_flag3 = . if wbcode == "PHL" | wbcode == "DOM" ;

/*************************************************************************************************************************/
/* Standardize the WVS measure of heterogeneity in political attitudes since it is based on an ordinal response variable */
/*************************************************************************************************************************/

egen std_wvs_polhet = std(wvs_leftright_sd) if smpl_flag3 == 0 ;

/**************************************************/
/* Run the regression for panel (a) of the figure */
/**************************************************/

reg lnnumgrps_fea03 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 0, vce(robust) noomitted vsquish ;

/************************************/
/* Generate panel (a) of the figure */
/************************************/

quietly { ;
  gen est_smpl = (e(sample) == 1) ;

  reg lnnumgrps_fea03 ${geovars} ${contall} if est_smpl == 1, vce(robust) ;
  predict resids1 if est_smpl == 1, resid ;

  reg pdiv_aa ${geovars} ${contall} if est_smpl == 1, vce(robust) ;
  predict resids2 if est_smpl == 1, resid ;

  reg resids1 resids2 if est_smpl == 1, vce(robust) ;
  local coef = string(_b[resids2], "%8.3f") ;
  local serr = string(_se[resids2], "%8.3f") ;
  local tsta = string(_b[resids2] / _se[resids2], "%8.3f") ;
  local ptr2 = string(`e(r2)', "%8.3f") ;
  local nobs = string(`e(N)', "%8.0g") ;
} ;

twoway (scatter resids1 resids2 if est_smpl == 1 & africa   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor1}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & europe   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor2}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & asia     == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor3}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & oceania  == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor4}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & namerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor5}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & samerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor6}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (lfit    resids1 resids2 if est_smpl == 1, clcolor(black) lstyle(refline) lwidth(medthick)),
        ylabel(, labsize(vsmall) nogrid)
        xlabel(, labsize(vsmall))
        ytitle("")
        xtitle("")
        l1title("(Residuals)" " ", size(vsmall))
        l2title("Log number of ethnic groups in the population", size(small))
        b1title(" " "(Residuals)", size(vsmall))
        b2title("Population diversity (ancestry adjusted)", size(small))
        note(" " "Relationship in the global sample; conditional on baseline geographical controls and continent fixed effects", span size(vsmall))
        caption("Slope coefficient = `coef'; (robust) standard error = `serr'; t-statistic = `tsta'; partial R-squared = `ptr2'; observations = `nobs'", span size(vsmall))
        legend(order(1 2 3 4 5 6) rows(1) label(1 "Africa") label(2 "Europe") label(3 "Asia") label(4 "Oceania") label(5 "N. America") label(6 "S. America") size(vsmall) position(6) region(style(none)))
        graphregion(color(white)) plotregion(color(white)) bgcolor(white) ;

graph export "../../results/figures/xfig_mechs_a.pdf", as(pdf) replace ;

capture drop est_smpl resids1 resids2 ;

/**************************************************/
/* Run the regression for panel (b) of the figure */
/**************************************************/

reg wvs_trust pdiv_aa ${geovars} ${regions} if smpl_flag2 == 0, vce(robust) noomitted vsquish ;

/************************************/
/* Generate panel (b) of the figure */
/************************************/

quietly { ;
  gen est_smpl = (e(sample) == 1) ;

  reg wvs_trust ${geovars} ${regions} if est_smpl == 1, vce(robust) ;
  predict resids1 if est_smpl == 1, resid ;

  reg pdiv_aa ${geovars} ${regions} if est_smpl == 1, vce(robust) ;
  predict resids2 if est_smpl == 1, resid ;

  reg resids1 resids2 if est_smpl == 1, vce(robust) ;
  local coef = string(_b[resids2], "%8.3f") ;
  local serr = string(_se[resids2], "%8.3f") ;
  local tsta = string(_b[resids2] / _se[resids2], "%8.3f") ;
  local ptr2 = string(`e(r2)', "%8.3f") ;
  local nobs = string(`e(N)', "%8.0g") ;
} ;

twoway (scatter resids1 resids2 if est_smpl == 1 & africa   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor1}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & europe   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor2}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & asia     == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor3}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & oceania  == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor4}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & namerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor5}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & samerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor6}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (lfit    resids1 resids2 if est_smpl == 1, clcolor(black) lstyle(refline) lwidth(medthick)),
        ylabel(, labsize(vsmall) nogrid)
        xlabel(, labsize(vsmall))
        ytitle("")
        xtitle("")
        l1title("(Residuals)" " ", size(vsmall))
        l2title("Prevalence of interpersonal trust in the population", size(small))
        b1title(" " "(Residuals)", size(vsmall))
        b2title("Population diversity (ancestry adjusted)", size(small))
        note(" " "Relationship in the global sample; conditional on baseline geographical controls and region fixed effects", span size(vsmall))
        caption("Slope coefficient = `coef'; (robust) standard error = `serr'; t-statistic = `tsta'; partial R-squared = `ptr2'; observations = `nobs'", span size(vsmall))
        legend(order(1 2 3 4 5 6) rows(1) label(1 "Africa") label(2 "Europe") label(3 "Asia") label(4 "Oceania") label(5 "N. America") label(6 "S. America") size(vsmall) position(6) region(style(none)))
        graphregion(color(white)) plotregion(color(white)) bgcolor(white) ;

graph export "../../results/figures/xfig_mechs_b.pdf", as(pdf) replace ;

capture drop est_smpl resids1 resids2 ;

/**************************************************/
/* Run the regression for panel (c) of the figure */
/**************************************************/

reg std_wvs_polhet pdiv_aa ${geovars} ${regions} if smpl_flag3 == 0, vce(robust) noomitted vsquish ;

/************************************/
/* Generate panel (c) of the figure */
/************************************/

quietly { ;
  gen est_smpl = (e(sample) == 1) ;

  reg std_wvs_polhet ${geovars} ${regions} if est_smpl == 1, vce(robust) ;
  predict resids1 if est_smpl == 1, resid ;

  reg pdiv_aa ${geovars} ${regions} if est_smpl == 1, vce(robust) ;
  predict resids2 if est_smpl == 1, resid ;

  reg resids1 resids2 if est_smpl == 1, vce(robust) ;
  local coef = string(_b[resids2], "%8.3f") ;
  local serr = string(_se[resids2], "%8.3f") ;
  local tsta = string(_b[resids2] / _se[resids2], "%8.3f") ;
  local ptr2 = string(`e(r2)', "%8.3f") ;
  local nobs = string(`e(N)', "%8.0g") ;
} ;

twoway (scatter resids1 resids2 if est_smpl == 1 & africa   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor1}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & europe   == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor2}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & asia     == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor3}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & oceania  == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor4}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & namerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor5}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (scatter resids1 resids2 if est_smpl == 1 & samerica == 1, msymbol(o) msize(medlarge) mfcolor("${RGBcolor6}") mlcolor(black) mlwidth(vthin) mlabel(wbcode) mlabpos(6) mlabsize(1.25) mlabcolor(gs8)) ||
       (lfit    resids1 resids2 if est_smpl == 1, clcolor(black) lstyle(refline) lwidth(medthick)),
        ylabel(, labsize(vsmall) nogrid)
        xlabel(, labsize(vsmall))
        ytitle("")
        xtitle("")
        l1title("(Residuals)" " ", size(vsmall))
        l2title("Variation in political attitudes in the population", size(small))
        b1title(" " "(Residuals)", size(vsmall))
        b2title("Population diversity (ancestry adjusted)", size(small))
        note(" " "Relationship in the global sample; conditional on baseline geographical controls and region fixed effects", span size(vsmall))
        caption("Slope coefficient = `coef'; (robust) standard error = `serr'; t-statistic = `tsta'; partial R-squared = `ptr2'; observations = `nobs'", span size(vsmall))
        legend(order(1 2 3 4 5 6) rows(1) label(1 "Africa") label(2 "Europe") label(3 "Asia") label(4 "Oceania") label(5 "N. America") label(6 "S. America") size(vsmall) position(6) region(style(none)))
        graphregion(color(white)) plotregion(color(white)) bgcolor(white) ;

graph export "../../results/figures/xfig_mechs_c.pdf", as(pdf) replace ;

capture drop est_smpl resids1 resids2 ;

/*******************************/
/* Close the log file and exit */
/*******************************/

log close ;

exit ;
