/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/****************************************************************/
/* FIGURE: Population Diversity and the Onset of Civil Conflict */
/****************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/**************************/
/* Specify a color scheme */
/**************************/

global RGBcolor1 "215 48 39" ;
global RGBcolor2 "252 141 89" ;
global RGBcolor3 "254 224 144" ;
global RGBcolor4 "224 243 248" ;
global RGBcolor5 "145 191 219" ;
global RGBcolor6 "69 117 180" ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/sfig_onset.log", text replace ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/*****************************************************************************/
/* Rescale some RHS variables to facilitate convergence of the MLE algorithm */
/*****************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

/*******************************************************************************************************/
/* Generate 1-year lags of the time-varying control, conflict incidence, and "time at peace" variables */
/*******************************************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_pri1 tap_a_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/**************************************************************************************************/
/* Generate cubic splines to control for the dependence of civil conflict onset on peace duration */
/**************************************************************************************************/

mkspline spline_pri1_ = tap_a_pri1_l1,  cubic nknots(4) ;

/*************************/
/* Generate year dummies */
/*************************/

xi, noomit i.year ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global splines "spline_pri1_1 spline_pri1_2 spline_pri1_3" ;
global timedum "_Iyear_1960-_Iyear_2016" ;

/**************************************************/
/* Run the regression for panel (a) of the figure */
/**************************************************/

probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${splines} ${timedum}
if year >= 1960 & year <= 2017 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/*****************************************************************************************************************************/
/* Compute the average marginal effect of diversity on the predicted likelihood of conflict onset for the current regression */
/*****************************************************************************************************************************/

margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish ;

matrix mfx_b = r(b) ;
matrix mfx_V = r(V) ;

local me_coef = mfx_b[1,1] ;
local me_serr = sqrt(mfx_V[1,1]) ;
local me_pval = 2 * (1 - normal(abs(`me_coef' / `me_serr'))) ;
local me_coef = string(`me_coef', "%8.3f") ;
local me_serr = string(`me_serr', "%8.3f") ;
local me_pval = string(`me_pval', "%8.3f") ;

matrix drop mfx_b mfx_V ;

/********************************************************************************************************/
/* Compute the vigintiles of the cross-country diversity distribution for the current estimation sample */
/********************************************************************************************************/

quietly { ;
  gen est_smpl = (e(sample) == 1) ;
  bysort est_smpl cid: gen cty_smpl = (_n == 1 & est_smpl == 1) ;

  centile pdiv_aa if cty_smpl == 1,  centile(5(5)95) ;

  capture drop est_smpl cty_smpl ;
} ;

/**********************************************************************************************************************/
/* Compute the predicted likelihood of conflict onset for every vigintile of the cross-country diversity distribution */
/**********************************************************************************************************************/

margins, predict(pr) at(pdiv_aa = (`r(c_1)'  `r(c_2)'  `r(c_3)'  `r(c_4)'  `r(c_5)'
                                               `r(c_6)'  `r(c_7)'  `r(c_8)'  `r(c_9)'  `r(c_10)'
                                               `r(c_11)' `r(c_12)' `r(c_13)' `r(c_14)' `r(c_15)'
                                               `r(c_16)' `r(c_17)' `r(c_18)' `r(c_19)')) vce(unconditional) vsquish noatlegend post ;

matrix t = J(19, 3, .) ;
matrix a = (5\10\15\20\25\30\35\40\45\50\55\60\65\70\75\80\85\90\95) ;

forvalues i = 1/19 { ;
  matrix t[`i',1] = _b[`i'._at] ;
  matrix t[`i',2] = _b[`i'._at] - 1.96 * _se[`i'._at] ;
  matrix t[`i',3] = _b[`i'._at] + 1.96 * _se[`i'._at] ;
} ;

matrix t = t, a ;
matrix colnames t = prob ll ul at ;
svmat t, names(col) ;

/************************************/
/* Generate panel (a) of the figure */
/************************************/

twoway (rarea ll ul at, color(gs14)) (line prob at, clcolor(black) lstyle(refline) lwidth(medthick)),
        ylabel(, labsize(vsmall) nogrid)
        xlabel(0(5)100, labsize(vsmall))
        ytitle("Predicted annual likelihood of new civil" "conflict onset, 1960-2017" " ", size(small))
        xtitle(" " "Percentile of cross-country population diversity distribution", size(small))
        note(" " "Predicted likelihoods based on a probit regression of conflict onset on diversity; conditional on all baseline controls", span size(vsmall))
        caption("Average marginal effect of a 0.01-increase in diversity = `me_coef' percent; standard error = `me_serr'; p-value = `me_pval'", span size(vsmall))
        legend(off)
		graphregion(color(white)) plotregion(color(white)) bgcolor(white) ;

graph export "../../results/figures/sfig_onset_a.pdf", as(pdf) replace ;

matrix drop t a ;

capture drop prob ll ul at ;

/**************************************************/
/* Run the regression for panel (b) of the figure */
/**************************************************/

ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis)
if year >= 1960 & year <= 2017, vce(cluster cid) first noomitted vsquish ;

/*****************************************************************************************************************************/
/* Compute the average marginal effect of diversity on the predicted likelihood of conflict onset for the current regression */
/*****************************************************************************************************************************/

margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish ;

matrix mfx_b = r(b) ;
matrix mfx_V = r(V) ;

local me_coef = mfx_b[1,1] ;
local me_serr = sqrt(mfx_V[1,1]) ;
local me_pval = 2 * (1 - normal(abs(`me_coef' / `me_serr'))) ;
local me_coef = string(`me_coef', "%8.3f") ;
local me_serr = string(`me_serr', "%8.3f") ;
local me_pval = string(`me_pval', "%8.3f") ;

matrix drop mfx_b mfx_V ;

/********************************************************************************************************/
/* Compute the vigintiles of the cross-country diversity distribution for the current estimation sample */
/********************************************************************************************************/

quietly { ;
  gen est_smpl = (e(sample) == 1) ;
  bysort est_smpl cid: gen cty_smpl = (_n == 1 & est_smpl == 1) ;

  centile pdiv_aa if cty_smpl == 1,  centile(5(5)95) ;

  capture drop est_smpl cty_smpl ;
} ;

/******************************************************************************************************************************/
/* Compute the predicted likelihood of conflict onset for every vigintile of the cross-country genetic diversity distribution */
/******************************************************************************************************************************/

margins, predict(pr) at(pdiv_aa = (`r(c_1)'  `r(c_2)'  `r(c_3)'  `r(c_4)'  `r(c_5)'
                                               `r(c_6)'  `r(c_7)'  `r(c_8)'  `r(c_9)'  `r(c_10)'
                                               `r(c_11)' `r(c_12)' `r(c_13)' `r(c_14)' `r(c_15)'
                                               `r(c_16)' `r(c_17)' `r(c_18)' `r(c_19)')) vce(unconditional) vsquish noatlegend post ;

matrix t = J(19, 3, .) ;
matrix a = (5\10\15\20\25\30\35\40\45\50\55\60\65\70\75\80\85\90\95) ;

forvalues i = 1/19 { ;
  matrix t[`i',1] = _b[`i'._at] ;
  matrix t[`i',2] = _b[`i'._at] - 1.96 * _se[`i'._at] ;
  matrix t[`i',3] = _b[`i'._at] + 1.96 * _se[`i'._at] ;
} ;

matrix t = t, a ;
matrix colnames t = prob ll ul at ;
svmat t, names(col) ;

/************************************/
/* Generate panel (b) of the figure */
/************************************/

twoway (rarea ll ul at, color(gs14)) (line prob at, clcolor(black) lstyle(refline) lwidth(medthick)),
        ylabel(, labsize(vsmall) nogrid)
        xlabel(0(5)100, labsize(vsmall))
        ytitle("Predicted annual likelihood of new civil" "conflict onset, 1960-2017" " ", size(small))
        xtitle(" " "Percentile of cross-country population diversity distribution", size(small))
        note(" " "Predicted likelihoods based on an IV probit regression of conflict onset on instrumented diversity; conditional on all baseline controls", span size(vsmall))
        caption("Average marginal effect of a 0.01-increase in diversity = `me_coef' percent; standard error = `me_serr'; p-value = `me_pval'", span size(vsmall))
        legend(off)
		graphregion(color(white)) plotregion(color(white)) bgcolor(white) ;

graph export "../../results/figures/sfig_onset_b.pdf", as(pdf) replace ;

matrix drop t a ;

capture drop prob ll ul at ;

/*******************************/
/* Close the log file and exit */
/*******************************/

log close ;

exit ;
