/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/***************************************************************************************/
/* TABLE: Population Diversity and the Count of Civil Conflict Onsets across Countries */
/***************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/xtab_count.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable evercol_gbr "Ever a U.K. colony dummy" ;
label variable evercol_fra "Ever a French colony dummy" ;
label variable evercol_oth "Ever a non-U.K./non-French colony dummy" ;
label variable legor_uk    "British legal origin dummy" ;
label variable legor_fr    "French legal origin dummy" ;

/* Contemporary institutional variables */
label variable xconst_1960_2017_avg_p4_v17 "Executive constraints, 1960--2017 average" ;
label variable democ_1960_2017_shr_p4_v17  "Fraction of years under democracy, 1960--2017" ;
label variable autoc_1960_2017_shr_p4_v17  "Fraction of years under autocracy, 1960--2017" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet                 "Oil or gas reserve discovery" ;
label variable lnpop_1960_2017_avg_wdi          "Log population, 1960--2017 average" ;
label variable lngdppc_cu_usd_1960_2017_avg_wdi "Log GDP per capita, 1960--2017 average" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "evercol_gbr evercol_fra evercol_oth" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_1960_2017_avg_p4_v17" ;
global polregs "democ_1960_2017_shr_p4_v17 autoc_1960_2017_shr_p4_v17" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_1960_2017_avg_wdi" ;
global ypercap "lngdppc_cu_usd_1960_2017_avg_wdi" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0) ;

qui egen smpl_flag2 = rowmiss(num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0) ;

/***********************/
/* Run the regressions */
/***********************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: nbreg num_ccst_60_17_sum_pri1 pdiv_aa if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col1_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col2_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col3_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col4_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col5_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${contold} if smpl_flag1 == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col6_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: nbreg num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if smpl_flag2 == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col7_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: poisson num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col8_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: poisson num_ccst_60_17_sum_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
qui margins, predict(n) dydx(pdiv_aa) vce(unconditional) vsquish post ;
eststo col9_me: nlcom (me_pdiv: _b[pdiv_aa] / 100), post ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/xtab_count.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa, relax) 
  indicate("Continent dummies=${contall}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N r2_p, fmt(%9.0f a2) labels("Observations" "Pseudo \$R^2\$") layout(@ @)) 
  varwidth(40) msign("\$-\$") label 
  prehead("\begin{tabular*}{680pt}{@{\extracolsep{\fill}}lccccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{5}{c}{Global}&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}\\" 
          "\cmidrule(r){2-6}\cmidrule(lr){7-8}\cmidrule(l){9-10}") 
  numbers mlabels("NegBin" "NegBin" "NegBin" "NegBin" "NegBin" "NegBin" "NegBin" "Poisson" "Poisson") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{9}{c}{Total count of new PRIO25 civil conflict onsets, 1960--2017}\\" 
           "\cmidrule{2-10}") 
  eqlabels("", none) varlabels(, elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me col9_me using "../../results/tables/xtab_count.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(40) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_pdiv "Marginal effect of diversity") 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
