/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/******************************************************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Incidence or Onset of Civil Conflict in Repeated Cross-Country Data - Robustness to Employing the Classical Logit and Rare-Events Logit Estimators */
/******************************************************************************************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/************************************************************************************************************/
/* Define a wrapper program for posting the marginal effect of diversity after rare-events logit estimation */
/************************************************************************************************************/

capture program drop relogitmfx ;
program define relogitmfx, eclass ;
  version 14.2 ;
  syntax [if], yvar(varname) xvars(varlist) ;

  qui relogit `yvar' `xvars' `if', cluster(cid) ;

  tempname div_1 div_2 ;
  qui sum pdiv_aa `if' ;
  scalar `div_1' = r(mean) ;
  scalar `div_2' = r(mean) + 0.01 ;
  setx mean `if' ;
  qui relogitq, fd(pr) changex(pdiv_aa `div_1' `div_2') ;

  tempname mfx_b mfx_V b V ;
  scalar `mfx_b' = r(dPr_1) * 100 ;
  scalar `mfx_V' = ((r(dPrU_1) - r(dPrL_1)) / (invnormal(0.975) * 2) * 100) ^ 2 ;
  matrix `b' = (`mfx_b') ;
  matrix `V' = (`mfx_V') ;
  matrix colnames `b' = pdiv_aa ;
  matrix rownames `b' = pdiv_aa ;
  matrix colnames `V' = pdiv_aa ;
  matrix rownames `V' = pdiv_aa ;

  ereturn post `b' `V' ;

  ereturn display ;
end ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_incst_logit.log", text replace ;

/************************************/
/* Load the AAGK 5-yr panel dataset */
/************************************/

use "../../data/aagk_5yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/*************************************************************************************/
/* Generate 1-period lags for the time-varying covariates and for conflict incidence */
/*************************************************************************************/

foreach var of varlist shrcol_gbr shrcol_fra shrcol_oth xconst_p4_v17 sdemoc_p4_v17 sautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_max_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.period ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable shrcol_gbr_l1 "Fraction of years as U.K. colony, lagged" ;
label variable shrcol_fra_l1 "Fraction of years as French colony, lagged" ;
label variable shrcol_oth_l1 "Fraction of years as other colony, lagged" ;
label variable legor_uk      "British legal origin dummy" ;
label variable legor_fr      "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Average executive constraints, lagged" ;
label variable sdemoc_p4_v17_l1 "Fraction of years under democracy, lagged" ;
label variable sautoc_p4_v17_l1 "Fraction of years under autocracy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log average population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log average GDP per capita, lagged" ;

/* Lagged conflict incidence */
label variable cconf_max_pri1_l1 "Civil conflict incidence, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "shrcol_gbr_l1 shrcol_fra_l1 shrcol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "sdemoc_p4_v17_l1 sautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global timedum "_Iperiod_30-_Iperiod_40" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & period >= 30 & period <= 41) ;

qui egen smpl_flag2 = rowmiss(cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & period >= 30 & period <= 41) ;

/**********************************************/
/* Run the regressions for conflict incidence */
/**********************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: logit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${contold} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Store some relevant information for the subsequent regression */
scalar N_clust = e(N_clust) ;
qui indeplist ; local rhs_lst `"`r(X)'"' ;

/* Obtain the marginal effect of diversity at the mean */
eststo col1_me: margins, predict(pr) dydx(pdiv_aa) atmeans vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: relogit cconf_max_pri1 `rhs_lst' if smpl_flag1 == 1 & oldw == 1, cluster(cid) ;

/* Record the number of countries in the sample (as per the preceding regression) */
estadd scalar N_clust : col2 ;

/* Obtain the marginal effect of diversity at the mean */
eststo col2_me: relogitmfx if e(sample) == 1, yvar(cconf_max_pri1) xvars(`rhs_lst') ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: logit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Store some relevant information for the subsequent regression */
scalar N_clust = e(N_clust) ;
qui indeplist ; local rhs_lst `"`r(X)'"' ;

/* Obtain the marginal effect of diversity at the mean */
eststo col3_me: margins, predict(pr) dydx(pdiv_aa) atmeans vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: relogit cconf_max_pri1 `rhs_lst' if smpl_flag2 == 1 & oldw == 1, cluster(cid) ;

/* Record the number of countries in the sample (as per the preceding regression) */
estadd scalar N_clust : col4 ;

/* Obtain the marginal effect of diversity at the mean */
eststo col4_me: relogitmfx if e(sample) == 1, yvar(cconf_max_pri1) xvars(`rhs_lst') ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/******************************************************************************************************/
/* Generate 1-year lags for the time-varying covariates and for conflict incidence and peace duration */
/******************************************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_pri1 tap_a_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/**************************************************************************************************/
/* Generate cubic splines to control for the dependence of civil conflict onset on peace duration */
/**************************************************************************************************/

mkspline spline_pri1_  = tap_a_pri1_l1, cubic nknots(4) ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.year ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable iscol_gbr_l1 "U.K. colony dummy, lagged" ;
label variable iscol_fra_l1 "French colony dummy, lagged" ;
label variable iscol_oth_l1 "Other colony dummy, lagged" ;
label variable legor_uk     "British legal origin dummy" ;
label variable legor_fr     "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Executive constraints, lagged" ;
label variable ddemoc_p4_v17_l1 "Democracy dummy, lagged" ;
label variable dautoc_p4_v17_l1 "Autocracy dummy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log GDP per capita, lagged" ;

/* Lagged conflict incidence */
label variable cconf_pri1_l1 "Civil conflict incidence, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global splines "spline_pri1_1 spline_pri1_2 spline_pri1_3" ;
global timedum "_Iyear_1960-_Iyear_2016" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & year >= 1960 & year <= 2017) ;

qui egen smpl_flag2 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & year >= 1960 & year <= 2017) ;

/******************************************/
/* Run the regressions for conflict onset */
/******************************************/

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: logit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${contold} ${splines} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Store some relevant information for the subsequent regression */
scalar N_clust = e(N_clust) ;
qui indeplist ; local rhs_lst `"`r(X)'"' ;

/* Obtain the marginal effect of diversity at the mean */
eststo col5_me: margins, predict(pr) dydx(pdiv_aa) atmeans vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: relogit ccst_pri1 `rhs_lst' if smpl_flag1 == 1 & oldw == 1, cluster(cid) ;

/* Record the number of countries in the sample (as per the preceding regression) */
estadd scalar N_clust : col6 ;

/* Obtain the marginal effect of diversity at the mean */
eststo col6_me: relogitmfx if e(sample) == 1, yvar(ccst_pri1) xvars(`rhs_lst') ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: logit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${splines} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Store some relevant information for the subsequent regression */
scalar N_clust = e(N_clust) ;
qui indeplist ; local rhs_lst `"`r(X)'"' ;

/* Obtain the marginal effect of diversity at the mean */
eststo col7_me: margins, predict(pr) dydx(pdiv_aa) atmeans vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: relogit ccst_pri1 `rhs_lst' if smpl_flag2 == 1 & oldw == 1, cluster(cid) ;

/* Record the number of countries in the sample (as per the preceding regression) */
estadd scalar N_clust : col8 ;

/* Obtain the marginal effect of diversity at the mean */
eststo col8_me: relogitmfx if e(sample) == 1, yvar(ccst_pri1) xvars(`rhs_lst') ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/stab_incst_logit.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa) 
  indicate("Continent dummies=${contold}" 
           "Time dummies=_Iperiod_* _Iyear_*" 
           "Controls for temporal spillovers=cconf_max_pri1_l1 cconf_pri1_l1 ${splines}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p, fmt(%9.0fc %9.0f a2) labels("Observations" "Countries" "Pseudo \$R^2\$") layout(@ @ @)) 
  varwidth(40) msign("\$-\$") label 
  prehead("\begin{tabular*}{695pt}{@{\extracolsep{\fill}}lcccccccc}" "\toprule") 
  numbers mlabels("Logit" "Logit" "Logit" "Logit" "Logit" "Logit" "Logit" "Logit") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{4}{c}{Quinquennial PRIO25 civil conflict}&\multicolumn{4}{c}{Annual PRIO25 civil conflict}\\" 
           "&\multicolumn{4}{c}{incidence, 1960--2017}&\multicolumn{4}{c}{onset, 1960--2017}\\" 
           "\cmidrule(r){2-5}\cmidrule(l){6-9}") 
  eqlabels("", none) varlabels(,elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me using "../../results/tables/stab_incst_logit.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(40) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity") 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
