/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*******************************************************************************************************************/
/* TABLE: Population Diversity and Intra-group Conflict Incidence in Cross-Country and Repeated Cross-Country Data */
/*******************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/mtab_intra.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable evercol_gbr "Ever a U.K. colony dummy" ;
label variable evercol_fra "Ever a French colony dummy" ;
label variable evercol_oth "Ever a non-U.K./non-French colony dummy" ;
label variable legor_uk    "British legal origin dummy" ;
label variable legor_fr    "French legal origin dummy" ;

/* Contemporary institutional variables */
label variable xconst_1985_2006_avg_p4_v17 "Executive constraints, 1985--2006 average" ;
label variable democ_1985_2006_shr_p4_v17  "Fraction of years under democracy, 1985--2006" ;
label variable autoc_1985_2006_shr_p4_v17  "Fraction of years under autocracy, 1985--2006" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet                 "Oil or gas reserve discovery" ;
label variable lnpop_1985_2006_avg_wdi          "Log population, 1985--2006 average" ;
label variable lngdppc_cu_usd_1985_2006_avg_wdi "Log GDP per capita, 1985--2006 average" ;

/* Minority group representation variables */
label variable numgrp_all_2001_amar    "Number of AMAR groups" ;
label variable gpro_AMAR_all_2001_amar "Population share of AMAR groups" ;

/*********************************************************************************************/
/* Create global macro lists with the variable names belonging to different sets of controls */
/*********************************************************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "evercol_gbr evercol_fra evercol_oth" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_1985_2006_avg_p4_v17" ;
global polregs "democ_1985_2006_shr_p4_v17 autoc_1985_2006_shr_p4_v17" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_1985_2006_avg_wdi" ;
global ypercap "lngdppc_cu_usd_1985_2006_avg_wdi" ;
global mingrps "numgrp_all_2001_amar gpro_AMAR_all_2001_amar" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;

/*****************************************************/
/* Generate the sample indicator for the regressions */
/*****************************************************/

qui egen smpl_flag = rowmiss(intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps}) ;
qui replace smpl_flag = (smpl_flag == 0) ;

/****************************************************************************************/
/* Run the regressions for the share of group-years with intra-group conflict (PANEL A) */
/****************************************************************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1A: reg intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${contold} if smpl_flag == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col1A ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col1A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${contold} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col1A ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2A: reg intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if smpl_flag == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col2A ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col2A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col2A ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3A: reg intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contold} if smpl_flag == 1 & oldw == 1, vce(robust) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col3A ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col3A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr intracgrpyr_shr_all_85_06_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contold} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col3A ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4A: ivregress 2sls intracgrpyr_shr_all_85_06_amar ${geovars} ${contall} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(robust) first noomitted vsquish ;

/* Store the first-stage statistics for subsequent use */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col4A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the first-stage regression coefficients and the first-stage F-statistic */
eststo col4A_fs: reg pdiv_aa mdist_addis ${geovars} ${contall} if smpl_flag == 1, vce(robust) noomitted vsquish ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col4A_fs ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5A: ivregress 2sls intracgrpyr_shr_all_85_06_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(robust) first noomitted vsquish ;

/* Store the first-stage statistics for subsequent use */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col5A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the first-stage regression coefficients and the first-stage F-statistic */
eststo col5A_fs: reg pdiv_aa mdist_addis ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag == 1, vce(robust) noomitted vsquish ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col5A_fs ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6A: ivregress 2sls intracgrpyr_shr_all_85_06_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contall} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(robust) first noomitted vsquish ;

/* Store the first-stage statistics for subsequent use */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col6A_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the first-stage regression coefficients and the first-stage F-statistic */
eststo col6A_fs: reg pdiv_aa mdist_addis ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contall} if smpl_flag == 1, vce(robust) noomitted vsquish ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col6A_fs ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1A col2A col3A col4A col5A col6A using "../../results/tables/mtab_intra.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa) 
  indicate("Continent dummies=${contall}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}" 
		   "Controls for number/share of AMAR groups=${mingrps}", labels("\$\times\$" " ")) 
  stats(N pc_pdiv adjr2, fmt(%9.0f a2 a2) labels("Observations" "Partial \$R^2\$ of population diversity" "Adjusted \$R^2\$") layout(@ @ @)) 
  varwidth(50) msign("\$-\$") label 
  prehead("\begin{tabular*}{680pt}{@{\extracolsep{\fill}}lcccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{3}{c}{Old World}&\multicolumn{3}{c}{Global}\\" 
          "\cmidrule(r){2-4}\cmidrule(l){5-7}") 
  numbers mlabels("OLS" "OLS" "OLS" "2SLS" "2SLS" "2SLS") collabels(none) 
  posthead("\midrule" 
           "PANEL~A&\multicolumn{6}{c}{Share of AMAR group-years with intra-group conflict, 1985--2006}\\" 
           "\cmidrule{2-7}") 
  varlabels(, elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1A_me col2A_me col3A_me col4A_me col5A_me col6A_me using "../../results/tables/mtab_intra.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(50) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_1090 "Effect of 10th--90th \%ile move in diversity") 
  postfoot("\midrule") ;

estout col4A_fs col5A_fs col6A_fs using "../../results/tables/mtab_intra.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(mdist_addis) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(50) msign("\$-\$") extracols(-2/0) label 
  mlabels(none) collabels(none) 
  posthead("FIRST STAGE&\multicolumn{3}{c}{}&\multicolumn{3}{c}{Population diversity (ancestry adjusted)}\\" 
           "\cmidrule{5-7}") 
  varlabels(,elist(mdist_addis \addlinespace)) 
  postfoot("\midrule\addlinespace") ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/***********************************************************************************/
/* Generate 1-year lags for the time-varying covariates and for conflict incidence */
/***********************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.year ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable iscol_gbr_l1 "U.K. colony dummy, lagged" ;
label variable iscol_fra_l1 "French colony dummy, lagged" ;
label variable iscol_oth_l1 "Other colony dummy, lagged" ;
label variable legor_uk     "British legal origin dummy" ;
label variable legor_fr     "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Executive constraints, lagged" ;
label variable ddemoc_p4_v17_l1 "Democracy dummy, lagged" ;
label variable dautoc_p4_v17_l1 "Autocracy dummy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log GDP per capita, lagged" ;

/* Minority group representation variables */
label variable numgrp_all_amar    "Number of AMAR groups" ;
label variable gpro_AMAR_all_amar "Population share of AMAR groups" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global mingrps "numgrp_all_amar gpro_AMAR_all_amar" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global timedum "_Iyear_1985-_Iyear_2005" ;

/*****************************************************/
/* Generate the sample indicator for the regressions */
/*****************************************************/

qui egen smpl_flag = rowmiss(intracon_max_all_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps}) ;
qui replace smpl_flag = (smpl_flag == 0 & year >= 1985 & year <= 2006) ;


/***************************************************************************/
/* Run the regressions for annual intra-group conflict incidence (PANEL B) */
/***************************************************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1B: probit intracon_max_all_amar pdiv_aa ${geovars} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col1B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;


/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2B: probit intracon_max_all_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col2B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3B: probit intracon_max_all_amar pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col3B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4B: ivprobit intracon_max_all_amar ${geovars} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col4B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls intracon_max_all_amar ${geovars} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col4B ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5B: ivprobit intracon_max_all_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col5B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls intracon_max_all_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col5B ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6B: ivprobit intracon_max_all_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col6B_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls intracon_max_all_amar ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${mingrps} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col6B ;

/*****************************************/
/* Append the results to the LaTeX table */
/*****************************************/

estout col1B col2B col3B col4B col5B col6B using "../../results/tables/mtab_intra.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) drop(pdiv_aa:*, relax) keep(pdiv_aa) 
  indicate("Controls as in same column of Panel A=${geovars}" 
           "Time dummies=_Iyear_*", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p, fmt(%9.0fc %9.0f a2) labels("Observations" "Countries" "Pseudo \$R^2\$") layout(@ @ @)) 
  varwidth(50) msign("\$-\$") label 
  mlabels("Probit" "Probit" "Probit" "IV Probit" "IV Probit" "IV Probit") collabels(none) 
  posthead("\midrule" 
           "PANEL~B&\multicolumn{6}{c}{Annual AMAR intra-group conflict incidence, 1985--2006}\\" 
           "\cmidrule{2-7}") 
  eqlabels("", none) varlabels(,elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1B_me col2B_me col3B_me col4B_me col5B_me col6B_me using "../../results/tables/mtab_intra.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(50) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity") 
  postfoot("\midrule") ;

estout col1B col2B col3B col4B col5B col6B using "../../results/tables/mtab_intra.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa:mdist_addis, relax) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(50) msign("\$-\$") label 
  mlabels(none) collabels(none)
  posthead("FIRST STAGE&\multicolumn{3}{c}{}&\multicolumn{3}{c}{Population diversity (ancestry adjusted)}\\"
           "\cmidrule{5-7}") 
  eqlabels("", none) varlabels(,elist(mdist_addis \addlinespace)) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
